#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import subprocess
import tkinter as tk
from tkinter import filedialog, messagebox

def choisir_fichier_source():
    fichier = filedialog.askopenfilename(filetypes=[("Fichiers PDF", "*.pdf")])
    if fichier:
        entree_source.delete(0, tk.END)
        entree_source.insert(0, fichier)

def choisir_fichier_cible():
    fichier = filedialog.asksaveasfilename(defaultextension=".pdf", filetypes=[("Fichiers PDF", "*.pdf")])
    if fichier:
        entree_cible.delete(0, tk.END)
        entree_cible.insert(0, fichier)

def get_page_count(pdf_file):
    try:
        output = subprocess.check_output(['pdfinfo', pdf_file], universal_newlines=True)
        for line in output.splitlines():
            if line.startswith('Pages:'):
                return int(line.split()[1])
    except Exception as e:
        messagebox.showerror("Erreur", f"Impossible de lire le nombre de pages :\n{e}")
        return None

def generer():
    fichier_source = entree_source.get().strip()
    fichier_cible = entree_cible.get().strip()
    mode = mode_var.get()  # "A3", "A4", ou "A4_2x1"

    # Vérification p (multiple de 4) — utile pour les modes livret
    if mode in ("A3", "A4"):
        try:
            p = int(entree_p.get())
            if p % 4 != 0:
                messagebox.showerror("Erreur", "Le nombre de pages par sujet (p) doit être un multiple de 4.")
                return
        except ValueError:
            messagebox.showerror("Erreur", "Veuillez saisir un nombre entier pour p.")
            return

    if not fichier_source or not fichier_cible:
        messagebox.showerror("Erreur", "Veuillez sélectionner un fichier source et un fichier cible.")
        return

    if not fichier_cible.lower().endswith(".pdf"):
        fichier_cible += ".pdf"

    total_pages = get_page_count(fichier_source)
    if total_pages is None:
        return


    try:
        if mode == "A3":
            # Mode livret A3 (imposition en signatures de p pages)
            cmd = [
                "pdfjam",
                "--booklet", "false",
                "--landscape",
                "--a3paper",
                "--signature", str(p),
                "--outfile", fichier_cible,
                fichier_source
            ]
        elif mode == "A4":
            # Mode livret A4 (imposition en signatures de p pages)
            cmd = [
                "pdfjam",
                "--booklet", "false",    # imposition livret 
                "--landscape",
                "--a4paper",
                "--signature", str(p),   # p multiple de 4 (4, 8, 12, …)
                "--outfile", fichier_cible,
                fichier_source
            ]
        else:
            # Correction A4 (2 pages par feuille, recto-verso) — non livret
            # Mode A4 paysage, 2 pages par face (ordre séquentiel : [1|2], [3|4], etc.)
            cmd = [
                "pdfjam",
                "--landscape",
                "--a4paper",
                "--nup", "2x1",
                "--outfile", fichier_cible,
                fichier_source
            ]

        subprocess.run(cmd, check=True)

        if mode == "A3":
            messagebox.showinfo(
                "Succès",
                "Le livret A3 a été généré avec succès.\n"
                "Imprimer en recto-verso (bords longs) sur A3."
            )
        elif mode == "A4":
            messagebox.showinfo(
                "Succès",
                "Le livret A4 a été généré avec succès.\n"
                "Imprimer en recto-verso (bords longs) sur A4."
            )
        else:
            messagebox.showinfo(
                "Succès",
                "La mise en page A4 (2 pages par feuille) a été générée avec succès.\n"
                "Imprimer en recto-verso (bords longs) sur A4."
            )

    except subprocess.CalledProcessError as e:
        messagebox.showerror("Erreur", f"Erreur lors de l'exécution de pdfjam :\n{e}")
    except FileNotFoundError:
        messagebox.showerror(
            "Erreur",
            "Commande introuvable. Vérifier que pdfjam et pdfinfo sont installés :\n"
            "sudo apt-get install texlive-extra-utils poppler-utils"
        )

# === Interface ===
root = tk.Tk()
root.title("util-jam — Livret A3 / A4 / Mise en page A4 (2 pages/feuille)")

# Fichier source
frame_source = tk.Frame(root)
frame_source.pack(padx=10, pady=5, fill='x')
tk.Label(frame_source, text="PDF source (pages A4 portrait) :").pack(side="left")
entree_source = tk.Entry(frame_source, width=54)
entree_source.pack(side="left", padx=6)
tk.Button(frame_source, text="Parcourir", command=choisir_fichier_source).pack(side="right")

# Fichier cible
frame_cible = tk.Frame(root)
frame_cible.pack(padx=10, pady=5, fill='x')
tk.Label(frame_cible, text="PDF de sortie :").pack(side="left")
entree_cible = tk.Entry(frame_cible, width=54)
entree_cible.pack(side="left", padx=6)
tk.Button(frame_cible, text="Parcourir", command=choisir_fichier_cible).pack(side="right")

# Choix du mode
frame_mode = tk.LabelFrame(root, text="Mode de composition")
frame_mode.pack(padx=10, pady=8, fill='x')
mode_var = tk.StringVar(value="A3")

# --- Ajout : fonction pour griser/dégriser l'entrée p selon le mode ---
def maj_etat_entree_p(*_):
    if mode_var.get() == "A4_2x1":
        entree_p.config(state="disabled")
    else:
        entree_p.config(state="normal")

tk.Radiobutton(frame_mode, text="Livret A3 (à partir de A4)", variable=mode_var, value="A3", command=maj_etat_entree_p).pack(anchor="w")
tk.Radiobutton(frame_mode, text="Livret A4 (à partir de A4)", variable=mode_var, value="A4", command=maj_etat_entree_p).pack(anchor="w")
tk.Radiobutton(frame_mode, text="Mise en page A4 (2 pages/feuille, recto-verso)", variable=mode_var, value="A4_2x1", command=maj_etat_entree_p).pack(anchor="w")

# p pages par signature
frame_p = tk.Frame(root)
frame_p.pack(padx=10, pady=5, fill='x')
tk.Label(frame_p, text="Pages par sujet (p, multiple de 4) :").pack(side="left")
entree_p = tk.Entry(frame_p, width=10)
entree_p.pack(side="left", padx=6)
entree_p.insert(0, "4")

# Bouton principal
tk.Button(root, text="Générer", command=generer, bg="#add8e6").pack(pady=12)

# Aide impression
help_lbl = tk.Label(
    root,
    text="Pour les livrets A3/A4 : imprimer en recto-verso « bords longs ».\n"
         "Pour la mise en page A4 (2 pages/feuille) : non-livret, ordre séquentiel ; imprimer en bords longs.",
    fg="gray"
)
help_lbl.pack(pady=(0,8))

# Signature
tk.Label(root, text="© José Ouin - https://www.joseouin.fr", font=("Arial", 9), fg="gray").pack(pady=(0, 6))

# --- Ajout : appliquer l'état initial en fonction du mode par défaut ---
maj_etat_entree_p()

root.mainloop()
